package org.msh.tb.bd.tbforms.dhis2;

import org.jboss.seam.Component;
import org.jboss.seam.contexts.Context;
import org.jboss.seam.contexts.Contexts;
import org.msh.tb.bd.tbforms.dhis2.data.DataValueResponse;
import org.msh.etbm.commons.transactionlog.DetailXMLWriter;
import org.msh.tb.application.TransactionManager;
import org.msh.tb.application.tasks.AsyncTaskImpl;
import org.msh.tb.application.tasks.TaskStatus;
import org.msh.tb.bd.Quarter;
import org.msh.tb.entities.*;
import org.msh.tb.entities.enums.RoleAction;
import org.msh.utils.JsonUtils;

import javax.persistence.EntityManager;
import java.util.Date;
import java.util.List;

/**
 * Created by Mauricio on 15/02/2017.
 */
public abstract class DHIS2ExportAsyncTask  extends AsyncTaskImpl {

    public static final String PARAM_TBUNITS = "tbunits";
    public static final String PARAM_SELECTED_QUARTER = "selectedQuarter";
    public static final String PARAM_DHIS2_AUTH_SERVICE = "dHIS2AuthenticationService";
    public static final String PARAM_DHIS2_EXPORT_SERVICE = "dHIS2DataExportService";

    /**
     * Units that will have its information sent to DHIS2
     */
    protected List<Tbunit> tbunits;

    /**
     * Quarter used to filter the information sent
     */
    protected Quarter selectedQuarter;

    protected DHIS2AuthenticationService dHIS2AuthenticationService;
    protected DHIS2DataExportService dHIS2DataExportService;
    protected TransactionManager transaction;
    protected Date taskStartingTime;

    /**
     * Returns the DHIS2 data set id
     * @return
     */
    protected abstract String getFormId();

    /**
     * Feed local class variables with values in params
     */
    @Override
    protected void starting() {
        this.setProgress(1);

        tbunits = (List<Tbunit>)getParameter(PARAM_TBUNITS);
        selectedQuarter = (Quarter)getParameter(PARAM_SELECTED_QUARTER);
        taskStartingTime = new Date();

        dHIS2AuthenticationService = (DHIS2AuthenticationService)getParameter(PARAM_DHIS2_AUTH_SERVICE);
        dHIS2DataExportService = (DHIS2DataExportService)getParameter(PARAM_DHIS2_EXPORT_SERVICE);
    }

    /**
     * Sets progress as 100
     */
    @Override
    protected void finishing() {
        if (getTransaction().isActive()) {
            getTransaction().commit();
        }

        this.setProgress(100);
    }

    /**
     * Must be called when DHIS2 returns the response for data sending.
     * It will register the transaction log of each tbunit integration.
     * @param response
     * @param tbunit
     * @param quarter
     */
    protected void saveTransactionLog(DataValueResponse response, Tbunit tbunit, Quarter quarter){
        if (!getTransaction().isActive()) {
            getTransaction().begin();
        }

        EntityManager em = getTransaction().getEntityManager();

        //avoid lazy init
        tbunit = em.find(Tbunit.class, tbunit.getId());

        UserLog userLog = em.find(UserLog.class, getUserLogId());
        WorkspaceLog workspaceLog = em.find(WorkspaceLog.class, tbunit.getWorkspace().getId());
        UserRole role = (UserRole) em.createQuery("from UserRole where name = :name")
                .setParameter("name", getEventName())
                .getResultList().get(0);
        tbunit = em.find(Tbunit.class, tbunit.getId());

        DetailXMLWriter detailWriter = new DetailXMLWriter();

        detailWriter.addTable();
        detailWriter.addTableRow("Exported Tb Unit", tbunit.getName() + " - " + tbunit.getAdminUnit().getFullDisplayName2());
        detailWriter.addTableRow("Exported Quarter", quarter.getDHIS2QuarterCode());
        detailWriter.addTableRow("Response Type", response.getResponseType());
        detailWriter.addTableRow("Status", response.getStatus());
        detailWriter.addTableRow("Description", response.getDescription());
        if (response.getImportCount() != null) {
            detailWriter.addTableRow("Imported", response.getImportCount().getImported());
            detailWriter.addTableRow("Updated", response.getImportCount().getUpdated());
            detailWriter.addTableRow("Ignored", response.getImportCount().getIgnored());
            detailWriter.addTableRow("Deleted", response.getImportCount().getDeleted());
        }
        detailWriter.addTableRow("Data Set Complete", response.isDataSetComplete());
        if (response.getConflicts() != null && !response.getConflicts().isEmpty()) {
            detailWriter.addTableRow("Conflicts", JsonUtils.objectToJSONString(response.getConflicts(), true));
        }

        // save the transaction log
        TransactionLog log = new TransactionLog();
        log.setAction(RoleAction.EXEC);
        log.setRole(role);
        log.setTransactionDate(getTaskStartingTime());
        log.setUser(userLog);
        log.setWorkspace(workspaceLog);
        log.setAdminUnit(tbunit.getAdminUnit());
        log.setUnit(tbunit);
        log.setComments(detailWriter.asXML());

        em.persist(log);

        getTransaction().commit();
    }

    /**
     * Returns the transaction log event name for the task
     * @return
     */
    protected abstract String getEventName();

    /**
     * Returns the date and time when the task just started
     * @return
     */
    protected Date getTaskStartingTime(){
        return taskStartingTime;
    }

    /**
     * Return the transaction in use by the task
     * @return
     */
    protected TransactionManager getTransaction() {
        if (transaction == null)
            transaction = (TransactionManager) Component.getInstance("transactionManager");
        return transaction;
    }

    public List<Tbunit> getTbunits() {
        return tbunits;
    }

    public void setTbunits(List<Tbunit> tbunits) {
        this.tbunits = tbunits;
    }

    public Quarter getSelectedQuarter() {
        return selectedQuarter;
    }

    public void setSelectedQuarter(Quarter selectedQuarter) {
        this.selectedQuarter = selectedQuarter;
    }

    public DHIS2AuthenticationService getdHIS2AuthenticationService() {
        return dHIS2AuthenticationService;
    }

    public void setdHIS2AuthenticationService(DHIS2AuthenticationService dHIS2AuthenticationService) {
        this.dHIS2AuthenticationService = dHIS2AuthenticationService;
    }

    public DHIS2DataExportService getdHIS2DataExportService() {
        return dHIS2DataExportService;
    }

    public void setdHIS2DataExportService(DHIS2DataExportService dHIS2DataExportService) {
        this.dHIS2DataExportService = dHIS2DataExportService;
    }

    public Integer getUserLogId(){
        return getUser().getId();
    }
}
